library(ggplot2)
library(stringr)
library(tidyverse)
library(haven)
library(readstata13)
library(cowplot)
library(ggsci)
library(summarytools)

plot_scale <- function(dat, y_val, file_name) {
  p <- ggplot(dat, aes(x=distance, y=y_val, color=the_scale)) + 
    geom_point(size=3) + 
    geom_smooth(method = "lm", size=1) +
    xlab("Scale") + ylab(file_name) +
    theme_cowplot(font_size=20) +
    theme(panel.background=element_blank(), 
          axis.line = element_line(colour = "black"),
          panel.grid.minor = element_blank(),
          legend.title=element_blank(), legend.position = c(0,.1)) +
    scale_color_lancet(labels=c("WEIRD Scale", "Sino Scale"))
  
  p 
  ggsave(paste0(file_name,".png"), p, width=6, height=5.5)
}

plot_scales <- function(dat1, dat2, y_val1, y_val2, file_name) {
  p <- ggplot() + 
    geom_point(data=dat1, aes(x=Cultural.Distance.x, y=y_val1), size=3, color="#00468BFF") + 
    geom_point(data=dat2, aes(x=Cultural.Distance.y, y=y_val2), size=3, color="#ED0000FF") + 
    geom_smooth(data=dat1, aes(x=Cultural.Distance.x, y=y_val1), color="#00468BFF", method = "lm", size=1) +
    geom_smooth(data=dat2, aes(x=Cultural.Distance.y, y=y_val2), color="#ED0000FF", method = "lm", size=1) +
    xlab("Scale") + ylab(file_name) +
    theme_cowplot(font_size=20) +
    theme(panel.background=element_blank(), 
          axis.line = element_line(colour = "black"),
          panel.grid.minor = element_blank(),
          legend.title=element_blank(), legend.position = c(0,.1)) +
    scale_color_lancet(labels=c("WEIRD Scale", "Sino Scale"))
  
  p 
  ggsave(paste0(file_name,".png"), p, width=6, height=5.5)
}

#Hofstede individualism with WEIRD scale----
hofstede <- read.csv("hofstede_cultural_dimensions.csv", na.strings = -1) #Scraped from Hofstede website using scrape_hofstede.py script
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Remove quotation marks and asterisk from Hofstede country names
hofstede$country <- gsub("'","", hofstede$country)
hofstede$country <- gsub("[*]","", hofstede$country)

#Merge
dat_hostede <- merge(x = hofstede, y = weird, by.x = 'country', by.y = 'Country', all = T)
dat_hostede <- merge(x = dat_hostede, y = sino, by.x = 'country', by.y = 'Country', all = T)


weird$Country <- as.character(weird$Country)
weird$Country[weird$Country=='Great Britain'] <- 'United Kingdom'
weird$Country[weird$Country=='Viet Nam'] <- 'Vietnam'

sino$Country <- as.character(sino$Country)
sino$Country[sino$Country=='Great Britain'] <- 'United Kingdom'
sino$Country[sino$Country=='Viet Nam'] <- 'Vietnam'

dat_hostede <- merge(x = hofstede, y = weird, by.x = 'country', by.y = 'Country', all = T)
dat_hostede <- merge(x = dat_hostede, y = sino, by.x = 'country', by.y = 'Country', all = T)

#Kogut-Singh1988
dat_hostede[dat_hostede$country=="United States",]
dat_hostede$ks.index.usa <- (((dat_hostede$pdi-40)^2)/var(dat_hostede$pdi,na.rm=T)+  
                                ((dat_hostede$idv-91)^2)/var(dat_hostede$idv,na.rm=T)+
                                ((dat_hostede$uai-46)^2)/var(dat_hostede$uai,na.rm=T)+
                                ((dat_hostede$mas-62)^2)/var(dat_hostede$mas,na.rm=T))/4

dat_hostede[dat_hostede$country=="China",]

dat_hostede$ks.index.china <- (((dat_hostede$pdi-80)^2)/var(dat_hostede$pdi,na.rm=T)+ 
                                  ((dat_hostede$idv-20)^2)/var(dat_hostede$idv,na.rm=T)+
                                  ((dat_hostede$uai-66)^2)/var(dat_hostede$uai,na.rm=T)+
                                  ((dat_hostede$mas-30)^2)/var(dat_hostede$mas,na.rm=T))/4

dat_hostede[dat_hostede$country=="United States",]
dat_hostede$ks.index.usa6 <- (((dat_hostede$pdi-40)^2)/var(dat_hostede$pdi,na.rm=T)+  
                               ((dat_hostede$idv-91)^2)/var(dat_hostede$idv,na.rm=T)+
                               ((dat_hostede$uai-46)^2)/var(dat_hostede$uai,na.rm=T)+
                               ((dat_hostede$lto-26)^2)/var(dat_hostede$lto,na.rm=T)+
                               ((dat_hostede$ind-68)^2)/var(dat_hostede$ind,na.rm=T)+
                               ((dat_hostede$mas-62)^2)/var(dat_hostede$mas,na.rm=T))/6

dat_hostede[dat_hostede$country=="China",]

dat_hostede$ks.index.china6 <- (((dat_hostede$pdi-80)^2)/var(dat_hostede$pdi,na.rm=T)+ 
                                 ((dat_hostede$idv-20)^2)/var(dat_hostede$idv,na.rm=T)+
                                 ((dat_hostede$uai-66)^2)/var(dat_hostede$uai,na.rm=T)+
                                  ((dat_hostede$lto-87)^2)/var(dat_hostede$lto,na.rm=T)+
                                  ((dat_hostede$ind-24)^2)/var(dat_hostede$ind,na.rm=T)+
                                 ((dat_hostede$mas-30)^2)/var(dat_hostede$mas,na.rm=T))/6

dat_hostede$pdi_usa <- abs(40-dat_hostede$pdi)
dat_hostede$idv_usa <- abs(91-dat_hostede$idv)
dat_hostede$uai_usa <- abs(46-dat_hostede$uai)
dat_hostede$lto_usa <- abs(26-dat_hostede$lto)
dat_hostede$ind_usa <- abs(68-dat_hostede$ind)
dat_hostede$mas_usa <- abs(62-dat_hostede$mas)

dat_hostede$pdi_china <- abs(80-dat_hostede$pdi)
dat_hostede$idv_china <- abs(20-dat_hostede$idv)
dat_hostede$uai_china <- abs(66-dat_hostede$uai)
dat_hostede$lto_china <- abs(87-dat_hostede$lto)
dat_hostede$ind_china <- abs(24-dat_hostede$ind)
dat_hostede$mas_china <- abs(30-dat_hostede$mas)

#Analysis
cor.test(dat_hostede$idv, dat_hostede$weird)
cor.test(dat_hostede$idv, dat_hostede$sino)
cor.test(dat_hostede$idv_usa, dat_hostede$weird)
cor.test(dat_hostede$idv_china, dat_hostede$sino)

cor.test(dat_hostede$pdi, dat_hostede$weird)
cor.test(dat_hostede$pdi, dat_hostede$sino)
cor.test(dat_hostede$pdi_usa, dat_hostede$weird)
cor.test(dat_hostede$pdi_china, dat_hostede$sino)

cor.test(dat_hostede$mas, dat_hostede$weird)
cor.test(dat_hostede$mas, dat_hostede$sino)
cor.test(dat_hostede$mas_usa, dat_hostede$weird)
cor.test(dat_hostede$mas_china, dat_hostede$sino)

cor.test(dat_hostede$uai, dat_hostede$weird)
cor.test(dat_hostede$uai, dat_hostede$sino)
cor.test(dat_hostede$uai_usa, dat_hostede$weird)
cor.test(dat_hostede$uai_china, dat_hostede$sino)

cor.test(dat_hostede$lto, dat_hostede$weird)
cor.test(dat_hostede$lto, dat_hostede$sino)
cor.test(dat_hostede$lto_usa, dat_hostede$weird)
cor.test(dat_hostede$lto_china, dat_hostede$sino)

cor.test(dat_hostede$ind, dat_hostede$weird)
cor.test(dat_hostede$ind, dat_hostede$sino)
cor.test(dat_hostede$ind_usa, dat_hostede$weird)
cor.test(dat_hostede$ind_china, dat_hostede$sino)

cor.test(dat_hostede$ks.index.usa, dat_hostede$weird)
cor.test(dat_hostede$ks.index.china, dat_hostede$sino)

cor.test(dat_hostede$ks.index.usa6, dat_hostede$weird)
cor.test(dat_hostede$ks.index.china6, dat_hostede$sino)


dat_hofstede_gathered <- gather(dat_hostede, "the_scale", "distance", c("weird","sino"))
dat_hofstede_gathered$the_scale <- factor(dat_hofstede_gathered$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$idv, "Individualism")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$idv_usa, dat_hostede$idv_china, 
            "Individualism Relative")

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$pdi, "Power Distance")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$pdi_usa, dat_hostede$pdi_china, 
            "Power Distance Relative")

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$mas, "Masculinity")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$mas_usa, dat_hostede$mas_china, 
            "Masculinity Relative")

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$uai, "Uncertainty Avoidance")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$uai_usa, dat_hostede$uai_china, 
            "Uncertainty Avoidance Relative")

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$lto, "Long Term Orientation")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$lto_usa, dat_hostede$lto_china, 
            "Long Term Orientation Relative")

plot_scale(dat_hofstede_gathered, dat_hofstede_gathered$ind, "Indulgence")
plot_scales(dat_hostede, dat_hostede, 
            dat_hostede$ind_usa, dat_hostede$ind_china, 
            "Indulgence Relative")

plot_scales(dat_hostede,dat_hostede, 
            dat_hostede$ks.index.usa, 
            dat_hostede$ks.index.china, 
            "Kogut-Singh Cultural Distance Original")

plot_scales(dat_hostede,dat_hostede, 
            dat_hostede$ks.index.usa6, 
            dat_hostede$ks.index.china6, 
            "Kogut-Singh Cultural Distance All")


#Tightness Gelfand Data----
gelfand <- read.csv("gelfand_tightness.csv")
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Merge
dat_gelfand <- merge(x = gelfand, y = weird, by.x = 'Nation', by.y = 'Country', all = T)
dat_gelfand$Country <- dat_gelfand$Nation
dat_gelfand$Country[is.na(dat_gelfand$Tightness.score)]
dat_gelfand$Country[is.na(dat_gelfand$Cultural.Distance)]

gelfand$Nation <- as.character(gelfand$Nation)
gelfand$Nation[gelfand$Nation=="People's Republic of China"] <- "China"
gelfand$Nation[gelfand$Nation=="United Kingdom"] <- "Great Britain"

dat_gelfand <- merge(x = gelfand, y = weird, by.x = 'Nation', by.y = 'Country', all = T)
dat_gelfand <- merge(x = dat_gelfand, y = sino, by.x = 'Nation', by.y = 'Country', all = T)


dat_gelfand$Country <- dat_gelfand$Nation

dat_gelfand$weird[dat_gelfand$Nation=="United States"] <- 0
dat_gelfand$sino[dat_gelfand$Nation=="China"] <- 0

dat_gelfand[dat_gelfand$Nation=="United States",]$Tightness.score #5.1
dat_gelfand[dat_gelfand$Nation=="China",]$Tightness.score #7.9

dat_gelfand$Tightness.score_usa <- abs(5.1-dat_gelfand$Tightness.score)
dat_gelfand$Tightness.score_china <- abs(7.9-dat_gelfand$Tightness.score)

#Analysis
cor.test(dat_gelfand$Tightness.score, dat_gelfand$weird)
cor.test(dat_gelfand$Tightness.score, dat_gelfand$sino)
cor.test(dat_gelfand$Tightness.score_usa, dat_gelfand$weird)
cor.test(dat_gelfand$Tightness.score_china, dat_gelfand$sino)

dat_gelfand_g <- gather(dat_gelfand, "the_scale", "distance", c("weird","sino"))
dat_gelfand_g$the_scale <- factor(dat_gelfand_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_gelfand_g, dat_gelfand_g$Tightness.score, "Tightness Gelfand")
plot_scales(dat_gelfand, dat_gelfand, 
            dat_gelfand$Tightness.score_usa,
            dat_gelfand$Tightness.score_china,
           "Tightness Gelfand Relative")


#Uz Tightness (lower is more tight) with WEIRD scale----
uz <- read.csv("uz_tightness.csv", na.strings = 'n/a')
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Merge
dat_uz <- merge(x = uz, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_uz$Country[is.na(dat_uz$CTL_C)]
dat_uz$Country[is.na(dat_uz$Cultural.Distance)]

uz$Country <- as.character(uz$Country)
uz$Country[uz$Country=="Egyptb"] <- "Egypt"
uz$Country[uz$Country=="Morrocob"] <- "Morroco"
uz$Country[uz$Country=="Netherland"] <- "Netherlands"
uz$Country[uz$Country=="Russian Federation"] <- "Russia"
uz$Country[uz$Country=="Serbia & Montenegro"] <- "Serbia and Montenegro"
uz$Country[uz$Country=="Swedena"] <- "Sweden"
uz$Country[uz$Country=="United States of America"] <- "United States"
weird$Country <- as.character(weird$Country)
weird$Country[weird$Country=='Viet Nam'] <- 'VietNam'
sino$Country <- as.character(sino$Country)
sino$Country[sino$Country=='Viet Nam'] <- 'VietNam'

dat_uz <- merge(x = uz, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_uz <- merge(x = dat_uz, y = sino, by.x = 'Country', by.y = 'Country', all = T)

dat_uz[dat_uz$Country=="United States",]$CTL_C #58
dat_uz[dat_uz$Country=="China",]$CTL_C #NA

dat_uz[dat_uz$Country=="United States",]$CTL_DS #67.9
dat_uz[dat_uz$Country=="China",]$CTL_DS #35.3

dat_uz[dat_uz$Country=="United States",]$CTL_DG #71.5
dat_uz[dat_uz$Country=="China",]$CTL_DG #NA

dat_uz$CTL_C_usa <- abs(58-dat_uz$CTL_C)
dat_uz$CTL_C_china <- abs(35.3-dat_uz$CTL_C)

dat_uz$CTL_DS_usa <- abs(67.9-dat_uz$CTL_DS)
dat_uz$CTL_DS_china <- abs(35.3-dat_uz$CTL_DS)

dat_uz$CTL_DG_usa <- abs(71.5-dat_uz$CTL_DG)
dat_uz$CTL_DG_china <- abs(35.3-dat_uz$CTL_DG)

#Analysis
cor.test(dat_uz$CTL_C, dat_uz$weird)
cor.test(dat_uz$CTL_C, dat_uz$sino)
cor.test(dat_uz$CTL_C_usa, dat_uz$weird)
cor.test(dat_uz$CTL_C_china, dat_uz$sino)

cor.test(dat_uz$CTL_DS, dat_uz$weird)
cor.test(dat_uz$CTL_DS, dat_uz$sino)
cor.test(dat_uz$CTL_DS_usa, dat_uz$weird)
cor.test(dat_uz$CTL_DS_china, dat_uz$sino)

cor.test(dat_uz$CTL_DG, dat_uz$weird)
cor.test(dat_uz$CTL_DG, dat_uz$sino)
cor.test(dat_uz$CTL_DG_usa, dat_uz$weird)
cor.test(dat_uz$CTL_DG_china, dat_uz$sino)

dat_uz[dat_uz$Country=="United States",]

dat_uz_g <- gather(dat_uz, "the_scale", "distance", c("weird","sino"))
dat_uz_g$the_scale <- factor(dat_uz_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_uz_g, dat_uz_g$CTL_C, "Looseness Composite")
plot_scales(dat_uz, dat_uz, 
            dat_uz$CTL_C_usa,
            dat_uz$CTL_C_china,
            "Looseness Composite Relative")

plot_scale(dat_uz_g, dat_uz_g$CTL_DS, "Looseness Domain Specific")
plot_scales(dat_uz, dat_uz, 
            dat_uz$CTL_DS_usa,
            dat_uz$CTL_DS_china,
            "Looseness Domain Specific Relative")

plot_scale(dat_uz_g, dat_uz_g$CTL_DG, "Looseness Domain General")
plot_scales(dat_uz, dat_uz, 
            dat_uz$CTL_DG_usa,
            dat_uz$CTL_DG_china,
            "Looseness Domain General Relative")



#Schwartz Values----
schwartz_culture <- read.csv("schwartz_culture_value_orientation.csv")
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

schwartz_culture$country <- str_to_title(schwartz_culture$country)
dat_schwartz <- merge(x = schwartz_culture, y = weird, by.x = 'country', by.y = 'Country', all = T)
dat_schwartz[is.na(dat_schwartz$aff.auton),]$country
dat_schwartz[is.na(dat_schwartz$Cultural.Distance),]$country

schwartz_culture$country[schwartz_culture$country=="Argentina     "] <- 'Argentina'
schwartz_culture$country[schwartz_culture$country=="Austrialia"] <- 'Australia'
schwartz_culture$country[schwartz_culture$country=="Canada (Eng)"] <- 'Canada'
schwartz_culture$country[schwartz_culture$country=="Hongkong"] <- 'Hong Kong'
schwartz_culture$country[schwartz_culture$country=="Korea South"] <- 'South Korea'
schwartz_culture$country[schwartz_culture$country=="Newzealand"] <- 'New Zealand'
schwartz_culture$country[schwartz_culture$country=="Serbia"] <- 'Serbia and Montenegro'
schwartz_culture$country[schwartz_culture$country=="Unitedkingdom"] <- 'Great Britain'
schwartz_culture$country[schwartz_culture$country=="Unitedstates"] <- 'United States'

dat_schwartz <- merge(x = schwartz_culture, y = weird, by.x = 'country', by.y = 'Country', all = T)
dat_schwartz <- merge(x = dat_schwartz, y = sino, by.x = 'country', by.y = 'Country', all = T)

dat_schwartz[dat_schwartz$country=="United States",]$harmony #3.46
dat_schwartz[dat_schwartz$country=="China",]$harmony #3.78
dat_schwartz$harmony_usa <- abs(3.46-dat_schwartz$harmony)
dat_schwartz$harmony_china <- abs(3.78-dat_schwartz$harmony)

dat_schwartz[dat_schwartz$country=="United States",]$embedded #3.67
dat_schwartz[dat_schwartz$country=="China",]$embedded #3.74
dat_schwartz$embedded_usa <- abs(3.67-dat_schwartz$embedded)
dat_schwartz$embedded_china <- abs(3.74-dat_schwartz$embedded)

dat_schwartz[dat_schwartz$country=="United States",]$hierarchy #2.37
dat_schwartz[dat_schwartz$country=="China",]$hierarchy #3.49
dat_schwartz$hierarchy_usa <- abs(2.37-dat_schwartz$hierarchy)
dat_schwartz$hierarchy_china <- abs(3.49-dat_schwartz$hierarchy)

dat_schwartz[dat_schwartz$country=="United States",]$mastery #4.09
dat_schwartz[dat_schwartz$country=="China",]$mastery #4.41
dat_schwartz$mastery_usa <- abs(4.09-dat_schwartz$mastery)
dat_schwartz$mastery_china <- abs(4.41-dat_schwartz$mastery)

dat_schwartz[dat_schwartz$country=="United States",]$aff.auton #3.87
dat_schwartz[dat_schwartz$country=="China",]$aff.auton #3.30
dat_schwartz$aff.auton_usa <- abs(3.87-dat_schwartz$aff.auton)
dat_schwartz$aff.auton_china <- abs(3.30-dat_schwartz$aff.auton)

dat_schwartz[dat_schwartz$country=="United States",]$intel.auton #4.19
dat_schwartz[dat_schwartz$country=="China",]$intel.auton #4.18
dat_schwartz$intel.auton_usa <- abs(4.19-dat_schwartz$intel.auton)
dat_schwartz$intel.auton_china <- abs(4.18-dat_schwartz$intel.auton)

dat_schwartz[dat_schwartz$country=="United States",]$egalitar #4.68
dat_schwartz[dat_schwartz$country=="China",]$egalitar #4.23
dat_schwartz$egalitar_usa <- abs(4.68-dat_schwartz$egalitar)
dat_schwartz$egalitar_china <- abs(4.23-dat_schwartz$egalitar)

cor.test(dat_schwartz$harmony, dat_schwartz$weird)
cor.test(dat_schwartz$harmony, dat_schwartz$sino)
cor.test(dat_schwartz$harmony_usa, dat_schwartz$weird)
cor.test(dat_schwartz$harmony_china, dat_schwartz$sino)

cor.test(dat_schwartz$mastery, dat_schwartz$weird)
cor.test(dat_schwartz$mastery, dat_schwartz$sino)
cor.test(dat_schwartz$mastery_usa, dat_schwartz$weird)
cor.test(dat_schwartz$mastery_china, dat_schwartz$sino)

cor.test(dat_schwartz$embedded, dat_schwartz$weird)
cor.test(dat_schwartz$embedded, dat_schwartz$sino)
cor.test(dat_schwartz$embedded_usa, dat_schwartz$weird)
cor.test(dat_schwartz$embedded_china, dat_schwartz$sino)

cor.test(dat_schwartz$hierarchy, dat_schwartz$weird)
cor.test(dat_schwartz$hierarchy, dat_schwartz$sino)
cor.test(dat_schwartz$hierarchy_usa, dat_schwartz$weird)
cor.test(dat_schwartz$hierarchy_china, dat_schwartz$sino)

cor.test(dat_schwartz$egalitar, dat_schwartz$weird)
cor.test(dat_schwartz$egalitar, dat_schwartz$sino)
cor.test(dat_schwartz$egalitar_usa, dat_schwartz$weird)
cor.test(dat_schwartz$egalitar_china, dat_schwartz$sino)

cor.test(dat_schwartz$aff.auton, dat_schwartz$weird)
cor.test(dat_schwartz$aff.auton, dat_schwartz$sino)
cor.test(dat_schwartz$aff.auton_usa, dat_schwartz$weird)
cor.test(dat_schwartz$aff.auton_china, dat_schwartz$sino)

cor.test(dat_schwartz$intel.auton, dat_schwartz$weird)
cor.test(dat_schwartz$intel.auton, dat_schwartz$sino)
cor.test(dat_schwartz$intel.auton_usa, dat_schwartz$weird)
cor.test(dat_schwartz$intel.auton_china, dat_schwartz$sino)


dat_schwartz_g <- gather(dat_schwartz, "the_scale", "distance", c("weird","sino"))
dat_schwartz_g$the_scale <- factor(dat_schwartz_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_schwartz_g, dat_schwartz_g$harmony, "Harmony")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$harmony_usa,
            dat_schwartz$harmony_china,
            "Harmony Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$mastery, "Mastery")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$mastery_usa,
            dat_schwartz$mastery_china,
            "Mastery Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$embedded, "Embeddedness")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$embedded_usa,
            dat_schwartz$embedded_china,
            "Embeddedness Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$hierarchy, "Hierarchy")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$hierarchy_usa,
            dat_schwartz$hierarchy_china,
            "Hierarchy Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$egalitar, "Egalitarianism")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$egalitar_usa,
            dat_schwartz$egalitar_china,
            "Egalitarianism Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$aff.auton, "Affective Autonomy")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$aff.auton_usa,
            dat_schwartz$aff.auton_china,
            "Affective Autonomy Relative")

plot_scale(dat_schwartz_g, dat_schwartz_g$intel.auton, "Intellectual Autonomy")
plot_scales(dat_schwartz, dat_schwartz, 
            dat_schwartz$intel.auton_usa,
            dat_schwartz$intel.auton_china,
            "Intellectual Autonomy Relative")

#Corruption Perceptions Index 2015----
cpi <- read.csv("2015_CPI_data.csv")
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Merge
dat_cpi <- merge(x = cpi, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_cpi$Country[is.na(dat_cpi$CPI2015)]
dat_cpi$Country[is.na(dat_cpi$Cultural.Distance)]

cpi$Country <- as.character(cpi$Country)
cpi$Country[cpi$Country=="United Kingdom"] <- "Great Britain"
cpi$Country[cpi$Country=="Vietnam"] <- "Viet Nam"
cpi$Country[cpi$Country=="Korea (South)"] <- "South Korea"
cpi$Country[cpi$Country=="The United States Of America"] <- "United States"

dat_cpi <- merge(x = cpi, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_cpi <- merge(x = dat_cpi, y = sino, by.x = 'Country', by.y = 'Country', all = T)


dat_cpi[dat_cpi$Country=="United States",]$CPI2015 #76
dat_cpi[dat_cpi$Country=="China",]$CPI2015 #37

dat_cpi$CPI_usa <- abs(76-dat_cpi$CPI2015)
dat_cpi$CPI_china <- abs(37-dat_cpi$CPI2015)

#Analysis
cor.test(dat_cpi$CPI2015, dat_cpi$weird)
cor.test(dat_cpi$CPI2015, dat_cpi$sino)
cor.test(dat_cpi$CPI_usa, dat_cpi$weird)
cor.test(dat_cpi$CPI_china, dat_cpi$sino)

dat_cpi_g <- gather(dat_cpi, "the_scale", "distance", c("weird","sino"))
dat_cpi_g$the_scale <- factor(dat_cpi_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_cpi_g, dat_cpi_g$CPI2015, "Corruption Perceptions Index")
plot_scales(dat_cpi, dat_cpi, 
            dat_cpi$CPI_usa,
            dat_cpi$CPI_china,
            "Corruption Perceptions Index Relative")


#Personality----
personality <- read.csv("McCrae2005Personality.csv")
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Merge
dat_big5 <- merge(x = personality, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_big5$Country[is.na(dat_big5$N)]
dat_big5$Country[is.na(dat_big5$Cultural.Distance)]
dat_big5 <- merge(x = dat_big5, y = sino, by.x = 'Country', by.y = 'Country', all = T)


dat_big5$O_usa <- abs(dat_big5[dat_big5$Country=="United States",]$O-dat_big5$O)
dat_big5$O_china <- abs(dat_big5[dat_big5$Country=="China",]$O-dat_big5$O)
dat_big5$C_usa <- abs(dat_big5[dat_big5$Country=="United States",]$C-dat_big5$C)
dat_big5$C_china <- abs(dat_big5[dat_big5$Country=="China",]$C-dat_big5$C)
dat_big5$E_usa <- abs(dat_big5[dat_big5$Country=="United States",]$E-dat_big5$E)
dat_big5$E_china <- abs(dat_big5[dat_big5$Country=="China",]$E-dat_big5$E)
dat_big5$A_usa <- abs(dat_big5[dat_big5$Country=="United States",]$A-dat_big5$A)
dat_big5$A_china <- abs(dat_big5[dat_big5$Country=="China",]$A-dat_big5$A)
dat_big5$N_usa <- abs(dat_big5[dat_big5$Country=="United States",]$N-dat_big5$N)
dat_big5$N_china <- abs(dat_big5[dat_big5$Country=="China",]$N-dat_big5$N)
dat_big5$SD_usa <- abs(dat_big5[dat_big5$Country=="United States",]$SD-dat_big5$SD)
dat_big5$SD_china <- abs(dat_big5[dat_big5$Country=="China",]$SD-dat_big5$SD)

#Analysis
cor.test(dat_big5$O, dat_big5$weird)
cor.test(dat_big5$O, dat_big5$sino)
cor.test(dat_big5$O_usa, dat_big5$weird)
cor.test(dat_big5$O_china, dat_big5$sino)

cor.test(dat_big5$C, dat_big5$weird)
cor.test(dat_big5$C, dat_big5$sino)
cor.test(dat_big5$C_usa, dat_big5$weird)
cor.test(dat_big5$C_china, dat_big5$sino)

cor.test(dat_big5$E, dat_big5$weird)
cor.test(dat_big5$E, dat_big5$sino)
cor.test(dat_big5$E_usa, dat_big5$weird)
cor.test(dat_big5$E_china, dat_big5$sino)

cor.test(dat_big5$A, dat_big5$weird)
cor.test(dat_big5$A, dat_big5$sino)
cor.test(dat_big5$A_usa, dat_big5$weird)
cor.test(dat_big5$A_china, dat_big5$sino)

cor.test(dat_big5$N, dat_big5$weird)
cor.test(dat_big5$N, dat_big5$sino)
cor.test(dat_big5$N_usa, dat_big5$weird)
cor.test(dat_big5$N_china, dat_big5$sino)

cor.test(dat_big5$SD, dat_big5$weird)
cor.test(dat_big5$SD, dat_big5$sino)
cor.test(dat_big5$SD_usa, dat_big5$weird)
cor.test(dat_big5$SD_china, dat_big5$sino)

dat_big5_g <- gather(dat_big5, "the_scale", "distance", c("weird","sino"))
dat_big5_g$the_scale <- factor(dat_big5_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_big5_g, dat_big5_g$O, "Openness")
plot_scales(dat_big5, dat_big5, 
            dat_big5$O_usa,
            dat_big5$O_china,
            "Openness Relative")

plot_scale(dat_big5_g, dat_big5_g$C, "Conscientiousness")
plot_scales(dat_big5, dat_big5, 
            dat_big5$C_usa,
            dat_big5$C_china,
            "Conscientiousness Relative")

plot_scale(dat_big5_g, dat_big5_g$E, "Extraversion")
plot_scales(dat_big5, dat_big5, 
            dat_big5$E_usa,
            dat_big5$E_china,
            "Extraversion Relative")

plot_scale(dat_big5_g, dat_big5_g$A, "Agreeableness")
plot_scales(dat_big5, dat_big5, 
            dat_big5$A_usa,
            dat_big5$A_china,
            "Agreeableness Relative")

plot_scale(dat_big5_g, dat_big5_g$N, "Neuroticism")
plot_scales(dat_big5, dat_big5, 
            dat_big5$N_usa,
            dat_big5$N_china,
            "Neuroticism Relative")

plot_scale(dat_big5_g, dat_big5_g$SD, "Personality Std. Dev")
plot_scales(dat_big5, dat_big5, 
            dat_big5$SD_usa,
            dat_big5$SD_china,
            "Personality Std. Dev Relative")

#CEPII Geographic and Linguistic Distance with WEIRD scale----
dist_cepii <- read.csv("dist_cepii.csv", na.strings = ".")
ling_dist <- read.dta13("ling_web.dta", missing.type = T)
ling_dist$lp1[ling_dist$lp1==0.0] <- NA
ling_dist$lp2[ling_dist$lp2==0.0] <- NA
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Restrict geo and language to USA and China
dist_cepii.usa <- dist_cepii[dist_cepii$iso_o=="USA",]
ling_dist.usa <- ling_dist[ling_dist$iso_o=="USA",]

dist_cepii.china <- dist_cepii[dist_cepii$iso_o=="CHN",]
ling_dist.china <- ling_dist[ling_dist$iso_o=="CHN",]

#Merge
dat_cepii.usa <- merge(x=dist_cepii.usa, y=ling_dist.usa, by="iso_d",all=T)
dat_cepii_weird <- merge(x = weird, y = dat_cepii.usa, by.x = 'Country', by.y = 'country_d', all = T)

dat_cepii.china <- merge(x=dist_cepii.china, y=ling_dist.china, by="iso_d",all=T)
dat_cepii_sino <- merge(x = sino, y = dat_cepii.china, by.x = 'Country', by.y = 'country_d', all = T)

dat_cepii_weird$Country[is.na(dat_cepii_weird$Cultural.Distance)]
dat_cepii_weird$Country[is.na(dat_cepii_weird$country_o)]

dat_cepii_sino$Country[is.na(dat_cepii_sino$Cultural.Distance)]
dat_cepii_sino$Country[is.na(dat_cepii_sino$country_o)]

dat_cepii.usa$country_d <- as.character(dat_cepii.usa$country_d)
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="United Kingdom of Great Britain and Northern Ireland"] <- "Great Britain"
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="Russian Federation"] <- "Russia"
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="Iran (Islamic Republic of)"] <- "Iran"
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="China, Hong Kong Special Administrative Region"] <- "Hong Kong"
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="Republic of Moldova"] <- "Moldova"
dat_cepii.usa$country_d[dat_cepii.usa$country_d=="Republic of Korea"] <- "South Korea"

dat_cepii.china$country_d <- as.character(dat_cepii.china$country_d)
dat_cepii.china$country_d[dat_cepii.china$country_d=="United Kingdom of Great Britain and Northern Ireland"] <- "Great Britain"
dat_cepii.china$country_d[dat_cepii.china$country_d=="Russian Federation"] <- "Russia"
dat_cepii.china$country_d[dat_cepii.china$country_d=="Iran (Islamic Republic of)"] <- "Iran"
dat_cepii.china$country_d[dat_cepii.china$country_d=="China, Hong Kong Special Administrative Region"] <- "Hong Kong"
dat_cepii.china$country_d[dat_cepii.china$country_d=="Republic of Moldova"] <- "Moldova"
dat_cepii.china$country_d[dat_cepii.china$country_d=="Republic of Korea"] <- "South Korea"
dat_cepii.china$country_d[dat_cepii.china$country_d=="United States of America"] <- "United States"

dat_cepii_weird <- merge(x = weird, y = dat_cepii.usa, by.x = 'Country', by.y = 'country_d', all = T)
dat_cepii_sino <- merge(x = sino, y = dat_cepii.china, by.x = 'Country', by.y = 'country_d', all = T)

#Analysis
cor.test(dat_cepii_weird$dist, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$dist, dat_cepii_sino$sino)

cor.test(dat_cepii_weird$distcap, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$distcap, dat_cepii_sino$sino)

cor.test(dat_cepii_weird$distw, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$distw, dat_cepii_sino$sino)

cor.test(dat_cepii_weird$distwces, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$distwces, dat_cepii_sino$sino)

cor.test(dat_cepii_weird$lp1, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$lp1, dat_cepii_sino$sino)

cor.test(dat_cepii_weird$lp2, dat_cepii_weird$weird)
cor.test(dat_cepii_sino$lp2, dat_cepii_sino$sino)

# dat_cepii_sino_weird <- merge(dat_cepii_weird, dat_cepii_sino, by = "Country", all = T)
# 
# dat_cepii_g <- gather(dat_cepii_sino_weird, "the_scale", "distance", c("weird","sino"))
# dat_cepii_g$the_scale <- factor(dat_cepii_g$the_scale, levels = c("weird", "sino")) 

dat_cepii_weird$distance <- dat_cepii_weird$weird
dat_cepii_sino$distance <- dat_cepii_sino$sino

dat_cepii_weird$Cultural.Distance.x <- dat_cepii_weird$weird
dat_cepii_sino$Cultural.Distance.y <- dat_cepii_sino$sino

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$dist, dat_cepii_sino$dist, "Geo Distance Pop Center")

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$distcap, dat_cepii_sino$distcap, "Geo Distance Capital")

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$distw, dat_cepii_sino$distw, "Geo Distance Gravity 1")

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$distwces, dat_cepii_sino$distwces, "Geo Distance Gravity 2")

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$lp1, dat_cepii_sino$lp1, "Language Distance Ethnologue")

plot_scales(dat_cepii_weird, dat_cepii_sino, dat_cepii_weird$lp2, dat_cepii_sino$lp2, "Language Distance ASJP")

#Genetic distance from USA with WEIRD scale----
gendist <- read.dta13("newgendist.dta")
weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Restrict comparison to USA
gendist.usa <- gendist[gendist$country_1=="U.S.A" | gendist$country_2=="U.S.A",]
gendist.usa$comp_country <- ifelse(gendist.usa$country_1=="U.S.A", gendist.usa$country_2, gendist.usa$country_1) #because USA can be in either country_1 or country_2, we need a separate variable for the comparison country

gendist.china <- gendist[gendist$country_1=="China" | gendist$country_2=="China",]
gendist.china$comp_country <- ifelse(gendist.china$country_1=="China", gendist.china$country_2, gendist.china$country_1) #because USA can be in either country_1 or country_2, we need a separate variable for the comparison country


#Merge
dat_gendist.usa <- merge(x = weird, y = gendist.usa, by.x = 'Country', by.y = 'comp_country', all = T)
dat_gendist.china <- merge(x = sino, y = gendist.china, by.x = 'Country', by.y = 'comp_country', all = T)

dat_gendist.usa$Country[is.na(dat_gendist.usa$wacziarg_1)]
dat_gendist.usa$Country[is.na(dat_gendist.usa$Cultural.Distance)]
dat_gendist.china$Country[is.na(dat_gendist.china$wacziarg_1)]
dat_gendist.china$Country[is.na(dat_gendist.china$Cultural.Distance)]

gendist.usa$comp_country <- as.character(gendist.usa$comp_country)
gendist.usa$comp_country[gendist.usa$comp_country=="U.S.S.R."] <- "Russia"
gendist.usa$comp_country[gendist.usa$comp_country=="Yemen, Arab Republic of"] <- "Yemen"
gendist.usa$comp_country[gendist.usa$comp_country=="Yemen, People's Democratic Republic of"] <- "Yemen"
gendist.usa$comp_country[gendist.usa$comp_country=="United Kingdom"] <- "Great Britain"
gendist.usa$comp_country[gendist.usa$comp_country=="Korea"] <- "South Korea"

gendist.china$comp_country <- as.character(gendist.china$comp_country)
gendist.china$comp_country[gendist.china$comp_country=="U.S.S.R."] <- "Russia"
gendist.china$comp_country[gendist.china$comp_country=="Yemen, Arab Republic of"] <- "Yemen"
gendist.china$comp_country[gendist.china$comp_country=="Yemen, People's Democratic Republic of"] <- "Yemen"
gendist.china$comp_country[gendist.china$comp_country=="United Kingdom"] <- "Great Britain"
gendist.china$comp_country[gendist.china$comp_country=="Korea"] <- "South Korea"
gendist.china$comp_country[gendist.china$comp_country=="U.S.A"] <- "United States"

weird$Country <- as.character(weird$Country)
weird$Country[weird$Country=='Viet Nam'] <- 'Vietnam'

sino$Country <- as.character(sino$Country)
sino$Country[sino$Country=='Viet Nam'] <- 'Vietnam'

dat_gendist.usa <- merge(x = weird, y = gendist.usa, by.x = 'Country', by.y = 'comp_country', all = T)
dat_gendist.china <- merge(x = sino, y = gendist.china, by.x = 'Country', by.y = 'comp_country', all = T)

#Analysis
cor.test(dat_gendist.usa$new_gendist_weighted, dat_gendist.usa$weird) #this is each ethnic group weighted
cor.test(dat_gendist.china$new_gendist_weighted, dat_gendist.china$sino) #this is each ethnic group weighted

cor.test(dat_gendist.usa$new_gendist_plurality, dat_gendist.usa$weird) #this is only comparing the plurality ethnic group
cor.test(dat_gendist.china$new_gendist_plurality, dat_gendist.china$sino) #this is only comparing the plurality ethnic group

dat_gendist.usa$distance <- dat_gendist.usa$weird
dat_gendist.china$distance <- dat_gendist.china$sino

dat_gendist.usa$Cultural.Distance.x <- dat_gendist.usa$weird
dat_gendist.china$Cultural.Distance.y <- dat_gendist.china$sino

plot_scales(dat_gendist.usa, dat_gendist.china, dat_gendist.usa$new_gendist_weighted, dat_gendist.china$new_gendist_weighted, "Genetic Distance Ethnic Weighting")

plot_scales(dat_gendist.usa, dat_gendist.china, dat_gendist.usa$new_gendist_plurality, dat_gendist.china$new_gendist_plurality, "Genetic Distance Plurality")

##PSYCHOLOGICAL MEASURES####
psych_dat <- read.dta13("DataMichael.dta")
dfSummary(psych_dat)

weird <- read.csv("weird_scale.csv") #Combining 2005 & 2010 Waves
sino <- read.csv("sino_scale.csv")

weird$weird <- weird$Cultural.Distance
sino$sino <- sino$Cultural.Distance

#Merge
dat_psych <- merge(x = psych_dat, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_psych$Country[is.na(dat_psych$fs_complexityIND_est)]
dat_psych$Country[is.na(dat_psych$Cultural.Distance)]

psych_dat$Country <- as.character(psych_dat$Country)
psych_dat$Country[psych_dat$Country=="Russian Federation"] <- "Russia"
psych_dat$Country[psych_dat$Country=="United States of America"] <- "United States"
psych_dat$Country[psych_dat$Country=="Hong Kong Special Administrative Region of China"] <- "Hong Kong"
psych_dat$Country[psych_dat$Country=="Republic of Korea"] <- "South Korea"
psych_dat$Country[psych_dat$Country=="Taiwan Province of China"] <- "Taiwan"
psych_dat$Country[psych_dat$Country=="Occupied Palestinian Territory"] <- "Palestine"
psych_dat$Country[psych_dat$Country=="Iran (Islamic Republic of)"] <- "Iran"


dat_psych <- merge(x = psych_dat, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_psych <- merge(x = dat_psych, y = sino, by.x = 'Country', by.y = 'Country', all = T)

#Creativity----
dat_psych[dat_psych$Country=="United States",]$wvs_dp_s_creative_C_std 
dat_psych[dat_psych$Country=="China",]$wvs_dp_s_creative_C_std  

dat_psych$creativity_usa <- abs(dat_psych[dat_psych$Country=="United States",]$wvs_dp_s_creative_C_std-dat_psych$wvs_dp_s_creative_C_std)
dat_psych$creativity_china <- abs(dat_psych[dat_psych$Country=="China",]$wvs_dp_s_creative_C_std-dat_psych$wvs_dp_s_creative_C_std)

#Analysis
cor.test(dat_psych$wvs_dp_s_creative_C_std, dat_psych$weird)
cor.test(dat_psych$wvs_dp_s_creative_C_std, dat_psych$sino)
cor.test(dat_psych$creativity_usa, dat_psych$weird)
cor.test(dat_psych$creativity_china, dat_psych$sino)

dat_psych_g <- gather(dat_psych, "the_scale", "distance", c("weird","sino"))
dat_psych_g$the_scale <- factor(dat_psych_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_psych_g, dat_psych_g$wvs_dp_s_creative_C_std, "Creativity")
plot_scales(dat_psych, dat_psych, 
            dat_psych$creativity_usa,
            dat_psych$creativity_china,
            "Creativity Relative")

#Blood Donations----
dat_psych[dat_psych$Country=="United States",]$BloodEstPopVol
dat_psych[dat_psych$Country=="China",]$BloodEstPopVol  

dat_psych$blood_usa <- abs(dat_psych[dat_psych$Country=="United States",]$BloodEstPopVol-dat_psych$BloodEstPopVol)
dat_psych$blood_china <- abs(dat_psych[dat_psych$Country=="China",]$BloodEstPopVol-dat_psych$BloodEstPopVol)

#Analysis
cor.test(dat_psych$BloodEstPopVol, dat_psych$weird)
cor.test(dat_psych$BloodEstPopVol, dat_psych$sino)
cor.test(dat_psych$blood_usa, dat_psych$weird)
cor.test(dat_psych$blood_china, dat_psych$sino)

dat_psych_g <- gather(dat_psych, "the_scale", "distance", c("weird","sino"))
dat_psych_g$the_scale <- factor(dat_psych_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_psych_g, dat_psych_g$BloodEstPopVol, "Blood Donation")
plot_scales(dat_psych, dat_psych, 
            dat_psych$blood_usa,
            dat_psych$blood_china,
            "Blood Donation Relative")

#Obedience----
dat_psych[dat_psych$Country=="United States",]$wvs_obedience
dat_psych[dat_psych$Country=="China",]$wvs_obedience  

dat_psych$obedience_usa <- abs(dat_psych[dat_psych$Country=="United States",]$wvs_obedience-dat_psych$wvs_obedience)
dat_psych$obedience_china <- abs(dat_psych[dat_psych$Country=="China",]$wvs_obedience-dat_psych$wvs_obedience)

#Analysis
cor.test(dat_psych$wvs_obedience, dat_psych$weird)
cor.test(dat_psych$wvs_obedience, dat_psych$sino)
cor.test(dat_psych$obedience_usa, dat_psych$weird)
cor.test(dat_psych$obedience_china, dat_psych$sino)

dat_psych_g <- gather(dat_psych, "the_scale", "distance", c("weird","sino"))
dat_psych_g$the_scale <- factor(dat_psych_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_psych_g, dat_psych_g$wvs_obedience, "Obedience Importance")
plot_scales(dat_psych, dat_psych, 
            dat_psych$obedience_usa,
            dat_psych$obedience_china,
            "Obedience Importance Relative")

#Parking Tickets----
dat_psych[dat_psych$Country=="United States",]$fm_lviol_pc
dat_psych[dat_psych$Country=="China",]$fm_lviol_pc  

dat_psych$parking_usa <- abs(dat_psych[dat_psych$Country=="United States",]$fm_lviol_pc-dat_psych$fm_lviol_pc)
dat_psych$parking_china <- abs(dat_psych[dat_psych$Country=="China",]$fm_lviol_pc-dat_psych$fm_lviol_pc)

#Analysis
cor.test(dat_psych$fm_lviol_pc, dat_psych$weird)
cor.test(dat_psych$fm_lviol_pc, dat_psych$sino)
cor.test(dat_psych$parking_usa, dat_psych$weird)
cor.test(dat_psych$parking_china, dat_psych$sino)

dat_psych_g <- gather(dat_psych, "the_scale", "distance", c("weird","sino"))
dat_psych_g$the_scale <- factor(dat_psych_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_psych_g, dat_psych_g$fm_lviol_pc, "Diplomat Parking Tickets")
plot_scales(dat_psych, dat_psych, 
            dat_psych$parking_usa,
            dat_psych$parking_china,
            "Diplomat Parking Tickets Relative")

#Honesty----
honesty_dat <- read.csv("honesty.csv")
#Merge
dat_honesty <- merge(x = honesty_dat, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_honesty$Country[is.na(dat_honesty$Honesty_Money)]
dat_honesty$Country[is.na(dat_honesty$Cultural.Distance)]

honesty_dat$Country <- as.character(honesty_dat$Country)
honesty_dat$Country[honesty_dat$Country=="Serbia"] <- "Serbia and Montenegro"

dat_honesty <- merge(x = honesty_dat, y = weird, by.x = 'Country', by.y = 'Country', all = T)
dat_honesty <- merge(x = dat_honesty, y = sino, by.x = 'Country', by.y = 'Country', all = T)

dat_honesty$Honesty_Money_usa <- abs(dat_honesty$Honesty_Money - 57.18519)
dat_honesty$Honesty_Money_china <- abs(dat_honesty$Honesty_Money - 21.48148)

dat_honesty$Honesty_NoMoney_usa <- abs(dat_honesty$Honesty_NoMoney - 38.814815)
dat_honesty$Honesty_NoMoney_china <- abs(dat_honesty$Honesty_NoMoney - 6.962963)

#Analysis
cor.test(dat_honesty$Honesty_Money, dat_honesty$weird)
cor.test(dat_honesty$Honesty_Money, dat_honesty$sino)
cor.test(dat_honesty$Honesty_Money_usa, dat_honesty$weird)
cor.test(dat_honesty$Honesty_Money_china, dat_honesty$sino)

dat_honesty_g <- gather(dat_honesty, "the_scale", "distance", c("weird","sino"))
dat_honesty_g$the_scale <- factor(dat_honesty_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_honesty_g, dat_honesty_g$Honesty_Money, "Return Wallet with Money")
plot_scales(dat_honesty, dat_honesty, 
            dat_honesty$Honesty_Money_usa,
            dat_honesty$Honesty_Money_china,
            "Return Wallet with Money Relative")

cor.test(dat_honesty$Honesty_NoMoney, dat_honesty$weird)
cor.test(dat_honesty$Honesty_NoMoney, dat_honesty$sino)
cor.test(dat_honesty$Honesty_NoMoney_usa, dat_honesty$weird)
cor.test(dat_honesty$Honesty_NoMoney_china, dat_honesty$sino)

dat_honesty_g <- gather(dat_honesty, "the_scale", "distance", c("weird","sino"))
dat_honesty_g$the_scale <- factor(dat_honesty_g$the_scale, levels = c("weird", "sino")) 

plot_scale(dat_honesty_g, dat_honesty_g$Honesty_NoMoney, "Return Wallet without Money")
plot_scales(dat_honesty, dat_honesty, 
            dat_honesty$Honesty_NoMoney_usa,
            dat_honesty$Honesty_NoMoney_china,
            "Return Wallet without Money Relative")